<?php

namespace App\Traits;

use App\Constants\Status;

trait RestaurantNotify
{
    public static function notifyToRestaurant()
    {
        return [
            'allRestaurants'              => 'All Restaurants',
            'selectedRestaurants'         => 'Selected Restaurants',
            'kycUnverified'         => 'Kyc Unverified Restaurants',
            'kycVerified'           => 'Kyc Verified Restaurants',
            'kycPending'            => 'Kyc Pending Restaurants',
            'withBalance'           => 'With Balance Restaurants',
            'emptyBalanceRestaurants'     => 'Empty Balance Restaurants',
            'twoFaDisableRestaurants'     => '2FA Disable Restaurant',
            'twoFaEnableRestaurants'      => '2FA Enable Restaurant',
            'hasDepositedRestaurants'       => 'Deposited Restaurants',
            'notDepositedRestaurants'       => 'Not Deposited Restaurants',
            'pendingDepositedRestaurants'   => 'Pending Deposited Restaurants',
            'rejectedDepositedRestaurants'  => 'Rejected Deposited Restaurants',
            'topDepositedRestaurants'     => 'Top Deposited Restaurants',
            'hasWithdrawRestaurants'      => 'Withdraw Restaurants',
            'pendingWithdrawRestaurants'  => 'Pending Withdraw Restaurants',
            'rejectedWithdrawRestaurants' => 'Rejected Withdraw Restaurants',
            'pendingTicketRestaurant'     => 'Pending Ticket Restaurants',
            'answerTicketRestaurant'      => 'Answer Ticket Restaurants',
            'closedTicketRestaurant'      => 'Closed Ticket Restaurants',
            'notLoginRestaurants'         => 'Last Few Days Not Login Restaurants',
        ];
    }

    public function scopeSelectedRestaurants($query)
    {
        return $query->whereIn('id', request()->restaurant ?? []);
    }

    public function scopeAllRestaurants($query)
    {
        return $query;
    }

    public function scopeEmptyBalanceRestaurants($query)
    {
        return $query->where('balance', '<=', 0);
    }

    public function scopeTwoFaDisableRestaurants($query)
    {
        return $query->where('ts', Status::DISABLE);
    }

    public function scopeTwoFaEnableRestaurants($query)
    {
        return $query->where('ts', Status::ENABLE);
    }

    public function scopeHasDepositedRestaurants($query)
    {
        return $query->whereHas('deposits', function ($deposit) {
            $deposit->successful();
        });
    }

    public function scopeNotDepositedRestaurants($query)
    {
        return $query->whereDoesntHave('deposits', function ($q) {
            $q->successful();
        });
    }

    public function scopePendingDepositedRestaurants($query)
    {
        return $query->whereHas('deposits', function ($deposit) {
            $deposit->pending();
        });
    }

    public function scopeRejectedDepositedRestaurants($query)
    {
        return $query->whereHas('deposits', function ($deposit) {
            $deposit->rejected();
        });
    }

    public function scopeTopDepositedRestaurants($query)
    {
        return $query->whereHas('deposits', function ($deposit) {
            $deposit->successful();
        })->withSum(['deposits' => function ($q) {
            $q->successful();
        }], 'amount')->orderBy('deposits_sum_amount', 'desc')->take(request()->number_of_top_deposited_restaurant ?? 10);
    }

    public function scopeHasWithdrawRestaurants($query)
    {
        return $query->whereHas('withdrawals', function ($q) {
            $q->approved();
        });
    }

    public function scopePendingWithdrawRestaurants($query)
    {
        return $query->whereHas('withdrawals', function ($q) {
            $q->pending();
        });
    }

    public function scopeRejectedWithdrawRestaurants($query)
    {
        return $query->whereHas('withdrawals', function ($q) {
            $q->rejected();
        });
    }

    public function scopePendingTicketRestaurant($query)
    {
        return $query->whereHas('tickets', function ($q) {
            $q->whereIn('status', [Status::TICKET_OPEN, Status::TICKET_REPLY]);
        });
    }

    public function scopeClosedTicketRestaurant($query)
    {
        return $query->whereHas('tickets', function ($q) {
            $q->where('status', Status::TICKET_CLOSE);
        });
    }

    public function scopeAnswerTicketRestaurant($query)
    {
        return $query->whereHas('tickets', function ($q) {

            $q->where('status', Status::TICKET_ANSWER);
        });
    }

    public function scopeNotLoginRestaurants($query)
    {
        return $query->whereDoesntHave('loginLogs', function ($q) {
            $q->whereDate('created_at', '>=', now()->subDays(request()->number_of_days ?? 10));
        });
    }

    public function scopeKycVerified($query)
    {
        return $query->where('kv', Status::KYC_VERIFIED);
    }
}
