<?php

namespace App\Models;

use App\Constants\Status;
use App\Traits\RestaurantNotify;
use App\Traits\SortOrder;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Carbon\Carbon;

class Restaurant extends Authenticatable
{
    use  RestaurantNotify, SortOrder;

    /**
     * The attributes that should be hidden for arrays.
     *
     * @var array
     */
    protected $hidden = [
        'password',
        'remember_token',
        'ver_code',
        'balance',
        'kyc_data',
    ];

    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
        'address' => 'object',
        'kyc_data' => 'object',
        'ver_code_send_at' => 'datetime',
    ];

    public function location()
    {
        return $this->belongsTo(Location::class);
    }

    public function loginLogs()
    {
        return $this->hasMany(UserLogin::class, 'restaurant_id');
    }

    public function transactions()
    {
        return $this->hasMany(Transaction::class)->orderBy('id', 'desc');
    }

    public function deposits()
    {
        return $this->hasMany(Deposit::class)->where('status', '!=', Status::PAYMENT_INITIATE);
    }

    public function withdrawals()
    {
        return $this->hasMany(Withdrawal::class)->where('status', '!=', Status::PAYMENT_INITIATE);
    }

    public function foods()
    {
        return $this->hasMany(Food::class);
    }

    public function tickets()
    {
        return $this->hasMany(SupportTicket::class);
    }

    public function plan()
    {
        return $this->belongsTo(Plan::class);
    }

    public function reviews()
    {
        return $this->hasMany(Review::class);
    }

    // SCOPES
    public function scopeActive($query)
    {
        return $query->where('status', Status::RESTAURANT_ACTIVE)->where('ev', Status::VERIFIED)->where('sv', Status::VERIFIED);
    }

    public function scopeBanned($query)
    {
        return $query->where('status', Status::RESTAURANT_BAN);
    }

    public function scopeEmailUnverified($query)
    {
        return $query->where('ev', Status::UNVERIFIED);
    }

    public function scopeMobileUnverified($query)
    {
        return $query->where('sv', Status::UNVERIFIED);
    }

    public function scopeKycUnverified($query)
    {
        return $query->where('kv', Status::KYC_UNVERIFIED);
    }

    public function scopeKycPending($query)
    {
        return $query->where('kv', Status::KYC_PENDING);
    }

    public function scopeEmailVerified($query)
    {
        return $query->where('ev', Status::VERIFIED);
    }

    public function scopeMobileVerified($query)
    {
        return $query->where('sv', Status::VERIFIED);
    }

    public function scopeWithBalance($query)
    {
        return $query->where('balance', '>', 0);
    }

    public function deviceTokens()
    {
        return $this->hasMany(DeviceToken::class);
    }

    public function followers()
    {
        return $this->belongsToMany(User::class, 'user_restaurant', 'restaurant_id', 'user_id')->withTimestamps();
    }

    public function featuredBadge(): Attribute
    {
        return new Attribute(function () {
            $html = '';
            if ($this->is_featured) {
                $html = '<span class="badge badge--success">' . trans('Yes') . '</span>';
            } else {
                $html = '<span class="badge badge--warning">' . trans('No') . '</span>';
            }
            return $html;
        });
    }

    public function averageRating()
    {
        $ratings = $this->reviews->pluck('rating');
        return ceil(collect($ratings)->avg());
    }

    public function reviewStars(): Attribute
    {
        $stars   = $this->averageRating();

        return new Attribute(function () use ($stars) {
            $html = '';
            for ($i = 0; $i < $stars; $i++) {
                $html .= '<i class="las la-star active"></i>';
            }
            for ($i = 0; $i < (5 - $stars); $i++) {
                $html .= '<i class="las la-star"></i>';
            }
            return $html;
        });
    }

    public function scopeNotExpiredPlans($query)
    {
        return $query->where(function ($query) {
            $query->whereNotNull('plan_id')
                ->whereNotNull('plan_purchased_at')
                ->whereHas('plan', function ($query) {
                    $query->whereColumn('restaurants.plan_id', 'plans.id')
                        ->where(function ($query) {
                            $query->whereRaw('restaurants.plan_purchased_at + INTERVAL plans.time DAY > NOW()');
                        });
                });
        });
    }

    public function scopeExpiredPlans($query)
    {
        return $query->where(function ($query) {
            $query->whereNotNull('plan_id')
                ->whereNotNull('plan_purchased_at')
                ->whereHas('plan', function ($query) {
                    $query->whereColumn('restaurants.plan_id', 'plans.id')
                        ->where(function ($query) {
                            $query->whereRaw('restaurants.plan_purchased_at + INTERVAL plans.time DAY <= NOW()');
                        });
                });
        });
    }

    public function planStatusBadge(): Attribute
    {
        return new Attribute(function () {
            if ($this->plan_id && $this->plan_purchased_at) {
                $plan = $this->plan;
                $planPurchasedAt = Carbon::parse($this->plan_purchased_at);
                $expirationTime = $planPurchasedAt->addDays($plan->time);

                if (Carbon::now()->greaterThanOrEqualTo($expirationTime)) {
                    // Plan is expired
                    return '<span class="badge badge--danger">' . trans('Expired') . '</span>';
                } else {
                    // Plan is active
                    return '<span class="badge badge--success">' . trans('Active') . '</span>';
                }
            } else {
                // No plan set
                return '<span class="badge badge--warning">' . trans('No Plan') . '</span>';
            }
        });
    }

    public function scopeFeatured($query)
    {
        return $query->where('is_featured', Status::YES);
    }
}
