<?php

namespace App\Models;

use App\Constants\Status;
use App\Traits\GlobalStatus;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Model;

class Order extends Model
{
    use GlobalStatus;

    public function items()
    {
        return $this->hasMany(OrderItem::class);
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function restaurant()
    {
        return $this->belongsTo(Restaurant::class);
    }

    public function address()
    {
        return $this->belongsTo(Address::class);
    }

    public function statusBadge(): Attribute
    {
        return new Attribute(function () {
            $html = '';
            if ($this->status == Status::ORDER_INITIATED) {
                $html = '<span class="badge badge--info">' . trans('Initiated') . '</span>';
            } elseif ($this->status == Status::ORDER_PROCESSING) {
                $html = '<span><span class="badge badge--secondary">' . trans('Processing') . '</span></span>';
            } elseif ($this->status == Status::ORDER_ON_DELIVERY) {
                $html = '<span><span class="badge badge--warning">' . trans('Out For Delivery') . '</span></span>';
            } elseif ($this->status == Status::ORDER_DELIVERED) {
                $html = '<span><span class="badge badge--success">' . trans('Delivered') . '</span></span>';
            } elseif ($this->status == Status::ORDER_CANCELLED) {
                $html = '<span><span class="badge badge--danger">' . trans('Cancelled') . '</span></span>';
            } elseif ($this->status == Status::ORDER_COMPLETED) {
                $html = '<span><span class="badge badge--success">' . trans('Completed') . '</span></span>';
            }
            return $html;
        });
    }

    public function typeBadge(): Attribute
    {
        return new Attribute(function () {
            $html = '';
            if ($this->payment_type == Status::PAYMENT_ONLINE_PAYMENT) {
                $html = '<span class="badge badge--info">' . trans('Online') . '</span>';
            } elseif ($this->payment_type == Status::PAYMENT_CASH_ON_DELIVERY) {
                $html = '<span><span class="badge badge--warning">' . trans('Cash on Delivery') . '</span></span>';
            }
            return $html;
        });
    }

    public function paymentTypeText(): Attribute
    {
        return new Attribute(function () {
            $text = '';
            if ($this->payment_type == Status::PAYMENT_ONLINE_PAYMENT) {
                $text = 'Online Payment';
            } elseif ($this->payment_type == Status::PAYMENT_CASH_ON_DELIVERY) {
                $text = 'Cash on Delivery';
            }
            return $text;
        });
    }

    public function paymentData(): Attribute
    {
        return new Attribute(function () {
            $html = '';
            if ($this->payment_status == STATUS::PAID && $this->status == Status::ORDER_CANCELLED) {
                $html = '<span class="badge badge--success">' . trans('Refunded') . '</span>';
            } elseif ($this->payment_status == Status::PAID) {
                $html = '<span class="badge badge--success">' . trans('Paid') . '</span>';
            } elseif ($this->payment_status == Status::UNPAID && $this->payment_type == Status::PAYMENT_CASH_ON_DELIVERY) {
                $html = '<span class="badge badge--success">' . trans('Cash on Delivery') . '</span>';
            } else {
                $html = '<span class="badge badge--warning">' . trans('Pending') . '</span>';
            }

            return $html;
        });
    }

    public function paymentStatusText(): Attribute
    {
        return new Attribute(function () {
            $text = '';
            if ($this->payment_status == STATUS::PAID && $this->status == Status::ORDER_CANCELLED) {
                $text = 'Refunded';
            } elseif ($this->payment_status == Status::PAID) {
                $text = 'Paid';
            } elseif ($this->payment_status == Status::UNPAID && $this->payment_type == Status::PAYMENT_CASH_ON_DELIVERY) {
                $text = 'Cash on Delivery';
            } else {
                $text = 'Payment Pending';
            }

            return $text;
        });
    }

    // scopes

    public function scopeUserOrders($query)
    {
        return $query->where('user_id', auth()->user()->id)->orderBy('id', 'DESC');
    }

    public function scopeRestaurantOrders($query)
    {
        return $query->where('restaurant_id', authRestaurantId())->orderBy('id', 'DESC');
    }

    public function scopeInitiated($query)
    {
        return $query->where('status', Status::ORDER_INITIATED);
    }

    public function scopeProcessing($query)
    {
        return $query->where('status', Status::ORDER_PROCESSING);
    }

    public function scopeOnDelivery($query)
    {
        return $query->where('status', Status::ORDER_ON_DELIVERY);
    }

    public function scopeDelivered($query)
    {
        return $query->where('status', Status::ORDER_DELIVERED);
    }

    public function scopeActive($query)
    {
        return $query->whereIn('status', [
            Status::ORDER_PROCESSING,
            Status::ORDER_ON_DELIVERY,
            Status::ORDER_DELIVERED,
        ])->orderBy('id', 'DESC');
    }

    public function scopeCompleted($query)
    {
        return $query->where('status', Status::ORDER_COMPLETED);
    }

    public function scopeCancelled($query)
    {
        return $query->where('status', Status::ORDER_CANCELLED);
    }

    public function scopePaid($query)
    {
        return $query->where('status', Status::PAID);
    }

    public function scopeUnpaid($query)
    {
        return $query->where('status', Status::UNPAID);
    }

    public function scopeOnlinePayment($query)
    {
        return $query->where('status', Status::PAYMENT_ONLINE_PAYMENT);
    }

    public function scopeCodPayment($query)
    {
        return $query->where('status', Status::PAYMENT_CASH_ON_DELIVERY);
    }

    public function scopePaymentPending($query)
    {
        return $query->where('payment_status', Status::UNPAID)->where('payment_type', Status::PAYMENT_ONLINE_PAYMENT)->orderBy('id', 'DESC');
    }

    public function scopeRefunded($query)
    {
        return $query->where('payment_status', Status::PAID)->where('status', Status::ORDER_CANCELLED)->orderBy('id', 'DESC');
    }
}
