<?php

namespace App\Http\Controllers\User;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Lib\FormProcessor;
use App\Lib\GoogleAuthenticator;
use App\Models\Address;
use App\Models\DeviceToken;
use App\Models\Form;
use App\Models\Order;
use App\Models\Restaurant;
use App\Models\Review;
use App\Models\Transaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class UserController extends Controller
{
    public function home()
    {
        $pageTitle = 'Dashboard';
        $user      = auth()->user();
        $orders    = Order::userOrders()->with('restaurant');

        $widget['balance']                = $user->balance;
        $widget['total_orders']           = (clone $orders)->count();
        $widget['active_orders']          = (clone $orders)->active()->count();
        $widget['payment_pending_orders'] = (clone $orders)->paymentPending()->count();
        $widget['favorite_restaurants']   = userInfo()['favorite_restaurant_count'];
        $widget['favorite_foods']         = $user->foods->count();
        $latestOrders                     = (clone $orders)->latest()->limit(5)->get();
        return view('Template::user.dashboard', compact('pageTitle', 'latestOrders', 'widget'));
    }

    public function depositHistory(Request $request)
    {
        $pageTitle = 'Deposit History';
        $deposits = auth()->user()->deposits()->searchable(['trx'])->with(['gateway'])->orderBy('id', 'desc')->paginate(getPaginate());
        return view('Template::user.deposit_history', compact('pageTitle', 'deposits'));
    }

    public function show2faForm()
    {
        $ga = new GoogleAuthenticator();
        $user = auth()->user();
        $secret = $ga->createSecret();
        $qrCodeUrl = $ga->getQRCodeGoogleUrl($user->username . '@' . gs('site_name'), $secret);
        $pageTitle = '2FA Security';
        return view('Template::user.twofactor', compact('pageTitle', 'secret', 'qrCodeUrl'));
    }

    public function create2fa(Request $request)
    {
        $user = auth()->user();
        $request->validate([
            'key' => 'required',
            'code' => 'required',
        ]);
        $response = verifyG2fa($user, $request->code, $request->key);
        if ($response) {
            $user->tsc = $request->key;
            $user->ts = Status::ENABLE;
            $user->save();
            $notify[] = ['success', 'Two factor authenticator activated successfully'];
            return back()->withNotify($notify);
        } else {
            $notify[] = ['error', 'Wrong verification code'];
            return back()->withNotify($notify);
        }
    }

    public function disable2fa(Request $request)
    {
        $request->validate([
            'code' => 'required',
        ]);

        $user = auth()->user();
        $response = verifyG2fa($user, $request->code);
        if ($response) {
            $user->tsc = null;
            $user->ts = Status::DISABLE;
            $user->save();
            $notify[] = ['success', 'Two factor authenticator deactivated successfully'];
        } else {
            $notify[] = ['error', 'Wrong verification code'];
        }
        return back()->withNotify($notify);
    }

    public function transactions()
    {
        $pageTitle = 'Transactions';
        $remarks = Transaction::distinct('remark')->orderBy('remark')->get('remark');

        $transactions = Transaction::where('user_id', auth()->id())->searchable(['trx'])->filter(['trx_type', 'remark'])->orderBy('id', 'desc')->paginate(getPaginate());

        return view('Template::user.transactions', compact('pageTitle', 'transactions', 'remarks'));
    }

    public function kycForm()
    {
        if (auth()->user()->kv == Status::KYC_PENDING) {
            $notify[] = ['error', 'Your KYC is under review'];
            return to_route('user.home')->withNotify($notify);
        }
        if (auth()->user()->kv == Status::KYC_VERIFIED) {
            $notify[] = ['error', 'You are already KYC verified'];
            return to_route('user.home')->withNotify($notify);
        }
        $pageTitle = 'KYC Form';
        $form = Form::where('act', 'user_kyc')->first();
        return view('Template::user.kyc.form', compact('pageTitle', 'form'));
    }

    public function kycData()
    {
        $user = auth()->user();
        $pageTitle = 'KYC Data';
        abort_if($user->kv == Status::VERIFIED, 403);
        return view('Template::user.kyc.info', compact('pageTitle', 'user'));
    }

    public function kycSubmit(Request $request)
    {
        $form = Form::where('act', 'user_kyc')->firstOrFail();
        $formData = $form->form_data;
        $formProcessor = new FormProcessor();
        $validationRule = $formProcessor->valueValidation($formData);
        $request->validate($validationRule);
        $user = auth()->user();
        foreach (@$user->kyc_data ?? [] as $kycData) {
            if ($kycData->type == 'file') {
                fileManager()->removeFile(getFilePath('verify') . '/' . $kycData->value);
            }
        }
        $userData = $formProcessor->processFormData($request, $formData);
        $user->kyc_data = $userData;
        $user->kyc_rejection_reason = null;
        $user->kv = Status::KYC_PENDING;
        $user->save();

        $notify[] = ['success', 'KYC data submitted successfully'];
        return to_route('user.home')->withNotify($notify);
    }

    public function userData()
    {
        $user = auth()->user();

        if ($user->profile_complete == Status::YES) {
            return to_route('user.home');
        }

        $pageTitle  = 'User Data';
        $info       = json_decode(json_encode(getIpInfo()), true);
        $mobileCode = @implode(',', $info['code']);
        $countries  = json_decode(file_get_contents(resource_path('views/partials/country.json')));

        return view('Template::user.user_data', compact('pageTitle', 'user', 'countries', 'mobileCode'));
    }

    public function userDataSubmit(Request $request)
    {

        $user = auth()->user();

        if ($user->profile_complete == Status::YES) {
            return to_route('user.home');
        }

        $countryData  = (array)json_decode(file_get_contents(resource_path('views/partials/country.json')));
        $countryCodes = implode(',', array_keys($countryData));
        $mobileCodes  = implode(',', array_column($countryData, 'dial_code'));
        $countries    = implode(',', array_column($countryData, 'country'));

        $request->validate([
            'country_code' => 'required|in:' . $countryCodes,
            'country'      => 'required|in:' . $countries,
            'mobile_code'  => 'required|in:' . $mobileCodes,
            'username'     => 'required|unique:users|min:6',
            'mobile'       => ['required', 'regex:/^([0-9]*)$/', Rule::unique('users')->where('dial_code', $request->mobile_code)],
        ]);


        if (preg_match("/[^a-z0-9_]/", trim($request->username))) {
            $notify[] = ['info', 'Username can contain only small letters, numbers and underscore.'];
            $notify[] = ['error', 'No special character, space or capital letters in username.'];
            return back()->withNotify($notify)->withInput($request->all());
        }

        $user->country_code = $request->country_code;
        $user->mobile       = $request->mobile;
        $user->username     = $request->username;


        $user->address = $request->address;
        $user->city = $request->city;
        $user->state = $request->state;
        $user->zip = $request->zip;
        $user->country_name = @$request->country;
        $user->dial_code = $request->mobile_code;

        $user->profile_complete = Status::YES;
        $user->save();

        return to_route('user.home');
    }


    public function addDeviceToken(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'token' => 'required',
        ]);

        if ($validator->fails()) {
            return ['success' => false, 'errors' => $validator->errors()->all()];
        }

        $deviceToken = DeviceToken::where('token', $request->token)->first();

        if ($deviceToken) {
            return ['success' => true, 'message' => 'Already exists'];
        }

        $deviceToken          = new DeviceToken();
        $deviceToken->user_id = auth()->user()->id;
        $deviceToken->token   = $request->token;
        $deviceToken->is_app  = Status::NO;
        $deviceToken->save();

        return ['success' => true, 'message' => 'Token saved successfully'];
    }

    public function downloadAttachment($fileHash)
    {
        $filePath = decrypt($fileHash);
        $extension = pathinfo($filePath, PATHINFO_EXTENSION);
        $title = slug(gs('site_name')) . '- attachments.' . $extension;
        try {
            $mimetype = mime_content_type($filePath);
        } catch (\Exception $e) {
            $notify[] = ['error', 'File does not exists'];
            return back()->withNotify($notify);
        }
        header('Content-Disposition: attachment; filename="' . $title);
        header("Content-Type: " . $mimetype);
        return readfile($filePath);
    }

    public function selectAddress($id)
    {
        $user    = auth()->user();
        $address = Address::where('user_id', $user->id)->findOrFail($id);
        $address->status = Status::ENABLE;
        $address->save();

        Address::where('user_id', $user->id)->where('id', '!=', $address->id)->update(['status' => Status::DISABLE]);

        return returnBack('Address selected successfully', 'success');
    }

    public function updateAddress(Request $request, $id = 0)
    {
        $request->validate([
            'apartment' => 'required',
            'street' => 'required',
            'type' => 'required|in:' . Status::USER_ADDRESS_HOME . ',' . Status::USER_ADDRESS_WORK . ',' . Status::USER_ADDRESS_OTHER . '',
        ]);

        $user = auth()->user();

        if ($id == 0) {
            $address = new Address();
        } else {
            $address = Address::where('user_id', $user->id)->findOrFail($id);
        }

        $address->user_id = $user->id;
        $address->apartment = $request->apartment;
        $address->street = $request->street;
        $address->type = $request->type;
        $address->status = Status::ENABLE;
        $address->save();
        Address::where('user_id', $user->id)->where('id', '!=', $address->id)->update(['status' => Status::DISABLE]);

        return returnBack('Address updated successfully', 'success');
    }

    public function deleteAddress(Request $request, $id)
    {
        $user = auth()->user();
        $address = Address::where('user_id', auth()->user()->id)->findOrFail($id);
        $address->delete();

        $address = Address::where('user_id', auth()->user()->id)->first();
        if ($address) {
            $address->status = Status::ENABLE;
            $address->save();
        }
        return returnBack('Address deleted successfully', 'success');
    }

    public function reviewRestaurant(Request $request)
    {
        $request->validate([
            'rating' => 'required|numeric|min:1|max:5',
            'review' => 'required|string',
            'restaurant_id' => 'required',
            'user_id' => 'required',
        ]);
        $user = auth()->user();
        $restaurant = Restaurant::active()->find($request->restaurant_id);

        if ($request->user_id !== $user->id) {
            returnBack('Invalid Request');
        }

        if (!$restaurant) {
            returnBack('Invalid Request');
        }

        $review = Review::where('user_id', $user->id)->where('restaurant_id', $restaurant->id)->first();

        if (!$review) {
            $review = new Review();
            $message = 'Review added successfully';
            $review->user_id = $request->user_id;
            $review->restaurant_id = $request->restaurant_id;
        } else {
            $message = 'Review updated successfully';
        }

        $review->review = $request->review;
        $review->rating = $request->rating;
        $review->save();

        return returnBack($message, 'success');
    }

    public function favoriteFoods()
    {
        $pageTitle   = "Favorite Foods";
        $user        = auth()->user();
        $foods       = $user->foods;
        $foods = customPaginate($foods, getPaginate());
        return view('Template::favorite_foods', compact('pageTitle', 'foods'));
    }
}
