<?php

namespace App\Http\Controllers\User;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Models\Addon;
use App\Models\Address;
use App\Models\Cart;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Restaurant;
use App\Models\Transaction;
use App\Models\VariantItem;
use Illuminate\Http\Request;

class OrderController extends Controller
{
    public function orders()
    {
        $pageTitle = 'All Orders';
        $orders    = $this->filterOrders();
        return view('Template::user.order.index', compact('pageTitle', 'orders'));
    }

    public function activeOrders()
    {
        $pageTitle = 'Active Orders';
        $orders    = $this->filterOrders('active');
        return view('Template::user.order.index', compact('pageTitle', 'orders'));
    }

    public function paymentPendingOrders()
    {
        $pageTitle = 'Payment Pending Orders';
        $orders    = $this->filterOrders('paymentPending');
        return view('Template::user.order.index', compact('pageTitle', 'orders'));
    }

    public function refundedOrders()
    {
        $pageTitle = 'Refunded Orders';
        $orders    = $this->filterOrders('refunded');
        return view('Template::user.order.index', compact('pageTitle', 'orders'));
    }

    protected function filterOrders($scope = null)
    {
        $orders = Order::userOrders();
        if ($scope) {
            $orders = $orders->$scope();
        }
        return $orders->with('restaurant')->searchable(['restaurant:restaurant_name'])->latest()->paginate(getPaginate());
    }

    public function orderDetails($orderId)
    {
        $pageTitle = 'Order Details';
        $order     = Order::userOrders()->with('restaurant', 'items', 'items.food')->findOrFail($orderId);
        return view('Template::user.order.details', compact('pageTitle', 'order'));
    }

    public function receivedOrder($id)
    {
        $order = Order::where('user_id', auth()->user()->id)
            ->whereIn('status', [
                Status::ORDER_ON_DELIVERY,
                Status::ORDER_DELIVERED,
            ])
            ->findOrFail($id);

        $order->status = Status::ORDER_COMPLETED;
        $order->save();
        $restaurant = $order->restaurant;
        $trx = getTrx();

        if ($order->payment_type == Status::PAYMENT_CASH_ON_DELIVERY) {
            //if cash on delivery substract the charge from restaurant balance
            $codCharge = $order->total_price * (gs('cod_percent_charge') / 100) + gs('cod_fixed_charge');
            $restaurant->balance -= $codCharge;
            $restaurant->save();

            $transaction                = new Transaction();
            $transaction->restaurant_id = $restaurant->id;
            $transaction->amount        = $codCharge;
            $transaction->post_balance  = $restaurant->balance;
            $transaction->charge        = 0;
            $transaction->trx_type      = '-';
            $transaction->details       = "Cash on Delivery charge";
            $transaction->trx           = $trx;
            $transaction->remark        = 'cod_charge';
            $transaction->save();
        } else {
            //pay to restaurant as user paid (online) through payment gateway
            $restaurant->balance += $order->total_price;
            $restaurant->save();

            $transaction                = new Transaction();
            $transaction->restaurant_id = $order->restaurant_id;
            $transaction->amount        = $order->total_price;
            $transaction->post_balance  = $restaurant->balance;
            $transaction->charge        = 0;
            $transaction->trx_type      = '+';
            $transaction->details       = "Payment Completed for order";
            $transaction->trx           = $trx;
            $transaction->remark        = 'order_completed';
            $transaction->save();
        }

        //notify restaurant and user
        notifyUser('user', 'ORDER_COMPLETED_USER', $order, $trx);
        notifyUser('restaurant', 'ORDER_COMPLETED_RESTAURANT', $order, $trx);

        return returnBack('Order received successfully', 'success');
    }

    public function checkout()
    {
        $pageTitle = "Checkout";
        $user      = auth()->user();
        $cartItems = Cart::where('user_id', $user->id)->with('user', 'restaurant', 'food')->get();
        if (count($cartItems) == 0) {
            return returnBack('Cart is empty', route: 'home');
        }

        $addresses = Address::where('user_id', $user->id)->get();
        return view('Template::user.checkout', compact('pageTitle', 'user', 'cartItems', 'addresses'));
    }

    public function submitOrder(Request $request)
    {
        $request->validate([
            'address_id'            => 'required|numeric',
            'payment_type'          => 'required|numeric',
            'additional_notes'      => 'nullable|string'
        ], [
            'address_id.required'   => 'Please add an address',
            'payment_type.required' => 'Please select a payment type',
        ]);

        $user = auth()->user();
        $cartItems = Cart::where('user_id', $user->id)->with('user', 'restaurant', 'food')->get();
        if (count($cartItems) == 0) {
            return returnBack('Cart is empty');
        }

        $address = Address::where('user_id', $user->id)->find($request->address_id);
        if (!$address) {
            return returnBack('Address not found');
        }

        $restaurant = $cartItems->first()->restaurant;
        $restaurant = Restaurant::active()->findOrFail($restaurant->id);

        //validate variants and addons
        foreach ($cartItems as $item) {
            if ($item->variations) {
                $variations = $item->variations;
                foreach ($variations as $variationItemId) {
                    $variationItem = VariantItem::where('id', $variationItemId)->first();
                    if (!$variationItem) {
                        return returnBack('Variation not found');
                    }
                    if (@$variationItem->variant->food->id != $item->food->id) {
                        return returnBack('Variation is invalid');
                    }
                }
            }
            if ($item->addons) {
                foreach ($item->addons as $addonId) {
                    $addonItem = Addon::where('id', $addonId)->first();
                    if (!$addonItem) {
                        return returnBack('Addon not found');
                    }
                    if (@$addonItem->food->id != $item->food->id) {
                        return returnBack('Addon is invalid');
                    }
                }
            }
        }

        //create a new order
        $order                   = new Order();
        $order->user_id          = $user->id;
        $order->restaurant_id    = $restaurant->id;
        $order->address          = $address->addressData();
        $order->additional_notes = $request->additional_notes;
        $order->payment_type     = $request->payment_type;
        $order->payment_status   = Status::UNPAID;
        $order->status           = Status::ORDER_INITIATED;
        $order->save();

        //calculate total price
        $orderTotalPrice = 0;
        $vat = $restaurant->vat;
        $deliveryFee = $restaurant->delivery_charge;

        foreach ($cartItems as $cartItem) {
            $orderItem = new OrderItem();
            $orderItem->order_id = $order->id;

            $totalFoodPrice = 0;
            $totalFoodPrice = $totalFoodPrice + $cartItem->food->price;
            if ($cartItem->variations) {
                $variations = $cartItem->variations;
                $variationsArray = [];
                foreach ($variations as $variationItemId) {
                    $variationItem     = VariantItem::find($variationItemId);
                    $totalFoodPrice    = $totalFoodPrice + $variationItem->price;
                    $variationsArray[] = [$variationItem->name, $variationItem->price];
                }
                $orderItem->variations = $variationsArray;
            }
            if ($cartItem->addons) {
                $addonsArray = [];
                foreach ($cartItem->addons as $addonId) {
                    $addonItem      = Addon::find($addonId);
                    $totalFoodPrice = $totalFoodPrice + $addonItem->price;
                    $addonsArray[]  = [$addonItem->name, $addonItem->price];
                }
                $orderItem->addons = $addonsArray;
            }

            $totalFoodPrice  = $totalFoodPrice * $cartItem->quantity;
            $orderTotalPrice = $orderTotalPrice + $totalFoodPrice;

            $orderItem->food_id          = $cartItem->food->id;
            $orderItem->item_price       = $cartItem->food->price;
            $orderItem->total_price      = $totalFoodPrice;
            $orderItem->additional_notes = $cartItem->additional_notes;
            $orderItem->quantity         = $cartItem->quantity;
            $orderItem->save();
        }

        $order->vat              = $orderTotalPrice * ($vat / 100);
        $order->delivery_charge  = $restaurant->delivery_charge;
        $order->total_price      = $orderTotalPrice + $deliveryFee + $order->vat;
        $order->save();

        //clear cart
        Cart::where('user_id', $user->id)->delete();

        notifyUser('user', 'ORDER_INITIATE', $order);

        if ($order->total_price <= 0 || $order->payment_type == Status::PAYMENT_CASH_ON_DELIVERY) {
            $order->status           = Status::ORDER_PROCESSING;
            $order->save();

            notifyUser('user', 'ORDER_PROCESSING_USER', $order);
            notifyUser('restaurant', 'ORDER_PROCESSING_RESTAURANT', $order);

            return returnBack('Order placed successfully', 'success', route: 'user.order.index');
        } else {
            $notify[] = ['success', 'Please make your payment'];
            return to_route('user.deposit.index', $order->id)->withNotify($notify);
        }
    }
}
