<?php

namespace App\Http\Controllers\Restaurant;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Lib\FormProcessor;
use App\Lib\GoogleAuthenticator;
use App\Models\Deposit;
use App\Models\Event;
use App\Models\Form;
use App\Models\Order;
use App\Models\Restaurant;
use App\Models\Review;
use App\Models\Transaction;
use App\Models\Withdrawal;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use App\Rules\FileTypeValidate;
use Illuminate\Support\Facades\Validator;

class RestaurantController extends Controller
{
    public function home()
    {
        $pageTitle  = 'Restaurant Dashboard';
        $restaurant = authRestaurant();
        $orders     = Order::restaurantOrders()->latest()->with('user');

        $widget['balance']                = $restaurant->balance;
        $widget['total_orders']           = (clone $orders)->count();
        $widget['active_orders']          = (clone $orders)->active()->count();
        $widget['processing_orders']      = (clone $orders)->processing()->count();
        $widget['onDelivery_orders']      = (clone $orders)->onDelivery()->count();
        $widget['delivered_orders']       = (clone $orders)->delivered()->count();
        $widget['cancelled_orders']       = (clone $orders)->cancelled()->count();
        $widget['pending_withdraws']      = Withdrawal::where('restaurant_id', $restaurant->id)->pending()->count();
        $latestOrders                     = (clone $orders)->latest()->limit(5)->get();

        //Sold for last 7 days
        $startDate = Carbon::now()->subDays(7)->startOfDay();
        $widget['sold_last_7_days'] = (clone $orders)->where('created_at', '>=', $startDate)->sum('total_price');

        //Sold for last 15 days
        $startDate = Carbon::now()->subDays(15)->startOfDay();
        $widget['sold_last_15_days'] = (clone $orders)->where('created_at', '>=', $startDate)->sum('total_price');

        //Sold for last 30 days
        $startDate = Carbon::now()->subDays(30)->startOfDay();
        $widget['sold_last_30_days'] = (clone $orders)->where('created_at', '>=', $startDate)->sum('total_price');

        $salesReport['date'] = collect([]);
        $sales = (clone $orders)
            ->where('created_at', '>=', Carbon::now()->subDays(30))
            ->selectRaw("SUM(total_price) as total_price, DATE_FORMAT(created_at,'%Y-%m-%d') as date")
            ->orderBy('created_at')
            ->groupBy('date')
            ->get();

        $sales->map(function ($trxData) use ($salesReport) {
            $salesReport['date']->push($trxData->date);
        });

        $salesReport['date'] = dateSorting($salesReport['date']->unique()->toArray());

        // Monthly Deposit & Withdraw Report Graph
        $report['months'] = collect([]);
        $report['deposit_month_amount'] = collect([]);
        $report['withdraw_month_amount'] = collect([]);

        $depositsMonth = Deposit::where('restaurant_id', authRestaurantId())->where('created_at', '>=', Carbon::now()->subYear())
            ->where('status', Status::PAYMENT_SUCCESS)
            ->selectRaw("SUM( CASE WHEN status = " . Status::PAYMENT_SUCCESS . " THEN amount END) as depositAmount")
            ->selectRaw("DATE_FORMAT(created_at,'%M-%Y') as months")
            ->orderBy('created_at')
            ->groupBy('months')->get();

        $depositsMonth->map(function ($depositData) use ($report) {
            $report['months']->push($depositData->months);
            $report['deposit_month_amount']->push(getAmount($depositData->depositAmount));
        });

        $withdrawalMonth = Withdrawal::where('restaurant_id', authRestaurantId())->where('created_at', '>=', Carbon::now()->subYear())->where('status', Status::PAYMENT_SUCCESS)
            ->selectRaw("SUM( CASE WHEN status = " . Status::PAYMENT_SUCCESS . " THEN amount END) as withdrawAmount")
            ->selectRaw("DATE_FORMAT(created_at,'%M-%Y') as months")
            ->orderBy('created_at')
            ->groupBy('months')->get();
        $withdrawalMonth->map(function ($withdrawData) use ($report) {
            if (!in_array($withdrawData->months, $report['months']->toArray())) {
                $report['months']->push($withdrawData->months);
            }
            $report['withdraw_month_amount']->push(getAmount($withdrawData->withdrawAmount));
        });

        $months = $report['months'];

        for ($i = 0; $i < $months->count(); ++$i) {
            $monthVal      = Carbon::parse($months[$i]);
            if (isset($months[$i + 1])) {
                $monthValNext = Carbon::parse($months[$i + 1]);
                if ($monthValNext < $monthVal) {
                    $temp = $months[$i];
                    $months[$i]   = Carbon::parse($months[$i + 1])->format('F-Y');
                    $months[$i + 1] = Carbon::parse($temp)->format('F-Y');
                } else {
                    $months[$i]   = Carbon::parse($months[$i])->format('F-Y');
                }
            }
        }

        return view('Template::restaurant.dashboard', compact('pageTitle', 'latestOrders', 'widget', 'salesReport', 'sales', 'depositsMonth', 'withdrawalMonth', 'months'));
    }

    public function depositHistory(Request $request)
    {
        $pageTitle = 'Deposit History';
        $deposits = authRestaurant()->deposits()->searchable(['trx'])->with(['gateway'])->orderBy('id', 'desc')->paginate(getPaginate());
        return view('Template::restaurant.deposit_history', compact('pageTitle', 'deposits'));
    }

    public function show2faForm()
    {
        $ga = new GoogleAuthenticator();
        $restaurant = authRestaurant();
        $secret = $ga->createSecret();
        $qrCodeUrl = $ga->getQRCodeGoogleUrl($restaurant->username . '@' . gs('site_name'), $secret);
        $pageTitle = '2FA Setting';
        return view('Template::restaurant.twofactor', compact('pageTitle', 'secret', 'qrCodeUrl'));
    }

    public function create2fa(Request $request)
    {
        $restaurant = authRestaurant();
        $request->validate([
            'key' => 'required',
            'code' => 'required',
        ]);
        $response = verifyG2fa($restaurant, $request->code, $request->key);
        if ($response) {
            $restaurant->tsc = $request->key;
            $restaurant->ts  = 1;
            $restaurant->save();
            $notify[] = ['success', 'Google authenticator activated successfully'];
            return back()->withNotify($notify);
        } else {
            $notify[] = ['error', 'Wrong verification code'];
            return back()->withNotify($notify);
        }
    }

    public function disable2fa(Request $request)
    {
        $request->validate([
            'code' => 'required',
        ]);

        $restaurant = authRestaurant();
        $response = verifyG2fa($restaurant, $request->code);
        if ($response) {
            $restaurant->tsc = null;
            $restaurant->ts  = 0;
            $restaurant->save();
            $notify[] = ['success', 'Two factor authenticator deactivated successfully'];
        } else {
            $notify[] = ['error', 'Wrong verification code'];
        }
        return back()->withNotify($notify);
    }

    public function transactions()
    {
        $pageTitle    = 'Transactions';
        $remarks      = Transaction::distinct('remark')->orderBy('remark')->get('remark');
        $transactions = Transaction::where('restaurant_id', authRestaurantId())->searchable(['trx'])->filter(['trx_type', 'remark'])->orderBy('id', 'desc')->paginate(getPaginate());
        return view('Template::restaurant.transactions', compact('pageTitle', 'transactions', 'remarks'));
    }

    public function kycForm()
    {
        if (authRestaurant()->kv == 2) {
            $notify[] = ['error', 'Your KYC is under review'];
            return to_route('restaurant.home')->withNotify($notify);
        }
        if (authRestaurant()->kv == 1) {
            $notify[] = ['error', 'You are already KYC verified'];
            return to_route('restaurant.home')->withNotify($notify);
        }
        $pageTitle = 'KYC Form';
        $form = Form::where('act', 'restaurant_kyc')->first();
        return view('Template::restaurant.kyc.form', compact('pageTitle', 'form'));
    }

    public function kycData()
    {
        $restaurant = authRestaurant();
        $pageTitle = 'KYC Data';
        return view('Template::restaurant.kyc.info', compact('pageTitle', 'restaurant'));
    }

    public function kycSubmit(Request $request)
    {
        $form                 = Form::where('act', 'restaurant_kyc')->first();
        $formData             = $form->form_data;
        $formProcessor        = new FormProcessor();
        $validationRule       = $formProcessor->valueValidation($formData);
        $request->validate($validationRule);
        $restaurantData       = $formProcessor->processFormData($request, $formData);
        $restaurant           = authRestaurant();
        $restaurant->kyc_data = $restaurantData;
        $restaurant->kv       = 2;
        $restaurant->save();

        $notify[] = ['success', 'KYC data submitted successfully'];
        return to_route('restaurant.home')->withNotify($notify);
    }

    public function downloadAttachment($fileHash)
    {
        $filePath  = decrypt($fileHash);
        $extension = pathinfo($filePath, PATHINFO_EXTENSION);
        $general   = gs();
        $title     = slug($general->site_name) . '- attachments.' . $extension;
        $mimetype  = mime_content_type($filePath);
        header('Content-Disposition: attachment; filename="' . $title);
        header("Content-Type: " . $mimetype);
        return readfile($filePath);
    }

    public function restaurantData()
    {
        $restaurant = authRestaurant();
        if ($restaurant->profile_complete == 1) {
            return to_route('restaurant.home');
        }
        $pageTitle = 'Restaurant Data';
        return view('Template::restaurant.restaurant_data', compact('pageTitle', 'restaurant'));
    }

    public function restaurantDataSubmit(Request $request)
    {
        $restaurant = authRestaurant();

        if ($restaurant->profile_complete == Status::YES) {
            return to_route('restaurant.home');
        }
        $request->validate([
            'restaurant_name'   => 'required|string|max:255',
            'profile_image'     => ['nullable', 'image', new FileTypeValidate(['jpg', 'jpeg', 'png'])],
            'cover_image'       => ['nullable', 'image', new FileTypeValidate(['jpg', 'jpeg', 'png'])],
            'cover_image_thumb' => ['nullable', 'image', new FileTypeValidate(['jpg', 'jpeg', 'png'])],
            'delivery_time'     => 'required|numeric|gt:0',
            'delivery_charge'   => 'required|numeric|gte:0',
            'username'          => 'required|unique:restaurants,username,' . $restaurant->id . '|min:6',
            'address'           => 'required|string|max:255',
            'state'             => 'required|string|max:255',
            'zip'               => 'required|string|max:20',
            'city'              => 'required|string|max:255',
        ]);

        $restaurant->restaurant_name = $request->restaurant_name;
        $restaurant->slug            = makeUniqueSlug($restaurant, $request->restaurant_name);
        $restaurant->delivery_time   = $request->delivery_time;
        $restaurant->delivery_charge = $request->delivery_charge;
        $restaurant->username        = $request->username;

        $restaurant->address = $request->address;
        $restaurant->city    = $request->city;
        $restaurant->state   = $request->state;
        $restaurant->zip     = $request->zip;

        if ($request->hasFile('profile_image')) {
            $restaurant->profile_image = fileUploader(
                $request->profile_image,
                getFilePath('restaurantProfile'),
                getFileSize('restaurantProfile'),
                $restaurant->profile_image,
                thumb: getFileThumbSize('restaurantProfile')
            );
        }

        if ($request->hasFile('cover_image')) {
            $restaurant->cover_image = fileUploader(
                $request->cover_image,
                getFilePath('restaurantCover'),
                getFileSize('restaurantCover'),
                $restaurant->cover_image
            );
        }

        if ($request->hasFile('cover_image_thumb')) {
            $restaurant->cover_image_thumb = fileUploader(
                $request->cover_image_thumb,
                getFilePath('restaurantCoverThumb'),
                getFileSize('restaurantCoverThumb'),
                $restaurant->cover_image_thumb
            );
        }

        $restaurant->profile_complete = Status::YES;
        $restaurant->save();

        return returnBack('Registration process completed successfully', 'success', route: 'restaurant.home');
    }

    public function reviews()
    {
        $pageTitle = 'My Reviews';
        $reviews   = Review::where('restaurant_id', authRestaurantId())->with('user')->paginate(getPaginate());
        return view('Template::restaurant.reviews', compact('pageTitle', 'reviews'));
    }
}
