<?php

namespace App\Http\Controllers\Restaurant;

use App\Http\Controllers\Controller;
use App\Models\Location;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules\Password;
use App\Rules\FileTypeValidate;
use Illuminate\Support\Facades\Validator;

class ProfileController extends Controller
{
    public function profile()
    {
        $pageTitle = "Profile Setting";
        $locations = Location::active()->orderBy('sort_order')->get();
        $restaurant = authRestaurant();
        return view('Template::restaurant.profile_setting', compact('pageTitle', 'restaurant', 'locations'));
    }

    public function submitProfile(Request $request)
    {
        $validation = Validator::make($request->all(), [
            'restaurant_name'      => 'required|string',
            'profile_image'        => ['nullable', 'image', new FileTypeValidate(['jpg', 'jpeg', 'png'])],
            'cover_image'          => ['nullable', 'image', new FileTypeValidate(['jpg', 'jpeg', 'png'])],
            'cover_image_thumb'    => ['nullable', 'image', new FileTypeValidate(['jpg', 'jpeg', 'png'])],
            'address'              => 'nullable|string|max:255',
            'city'                 => 'nullable|string|max:100',
            'state'                => 'nullable|string|max:100',
            'zip'                  => 'nullable|string|max:20',
            'location'             => 'required|exists:locations,id',
        ], [
            'restaurant_name.required' => 'The restaurant name field is required.',
            'restaurant_name.string'   => 'The restaurant name must be a string.',
            'profile_image.image'      => 'The profile image must be an image file.',
            'cover_image.image'        => 'The cover image must be an image file.',
            'cover_image_thumb.image'  => 'The cover image thumbnail must be an image file.',
            'address.max'              => 'The address may not be greater than 255 characters.',
            'city.max'                 => 'The city may not be greater than 100 characters.',
            'state.max'                => 'The state may not be greater than 100 characters.',
            'zip.max'                  => 'The zip code may not be greater than 20 characters.',
            'location.required'        => 'The location field is required.',
            'location.exists'          => 'The selected location is invalid.',
        ]);

        if ($validation->fails()) {
            return back()->withErrors($validation)->withInput();
        }

        $restaurant = authRestaurant();

        $restaurant->address = $request->address;
        $restaurant->city = $request->city;
        $restaurant->state = $request->state;
        $restaurant->zip = $request->zip;
        $restaurant->location_id = $request->location;

        if ($request->profile_image) {
            $restaurant->profile_image = fileUploader($request->profile_image, getFilePath('restaurantProfile'), getFileSize('restaurantProfile'), $restaurant->profile_image, thumb: getFileThumbSize('restaurantProfile'));
        }
        if ($request->cover_image) {
            $restaurant->cover_image = fileUploader($request->cover_image, getFilePath('restaurantCover'), getFileSize('restaurantCover'), $restaurant->cover_image);
        }
        if ($request->cover_image_thumb) {
            $restaurant->cover_image_thumb = fileUploader($request->cover_image_thumb, getFilePath('restaurantCoverThumb'), getFileSize('restaurantCoverThumb'), $restaurant->cover_image_thumb);
        }

        $restaurant->restaurant_name = $request->restaurant_name;
        $restaurant->slug            = makeUniqueSlug($restaurant, $request->restaurant_name);
        $restaurant->save();

        $notify[] = ['success', 'Profile updated successfully'];
        return back()->withNotify($notify);
    }


    public function changePassword()
    {
        $pageTitle = 'Change Password';
        return view('Template::restaurant.password', compact('pageTitle'));
    }

    public function submitPassword(Request $request)
    {

        $passwordValidation = Password::min(6);
        if (gs('secure_password')) {
            $passwordValidation = $passwordValidation->mixedCase()->numbers()->symbols()->uncompromised();
        }

        $request->validate([
            'current_password' => 'required',
            'password' => ['required', 'confirmed', $passwordValidation]
        ]);

        $restaurant = authRestaurant();
        if (Hash::check($request->current_password, $restaurant->password)) {
            $password = Hash::make($request->password);
            $restaurant->password = $password;
            $restaurant->save();
            $notify[] = ['success', 'Password changed successfully'];
            return back()->withNotify($notify);
        } else {
            $notify[] = ['error', 'The password doesn\'t match!'];
            return back()->withNotify($notify);
        }
    }
}
