<?php

namespace App\Http\Controllers\Restaurant;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Models\Addon;
use App\Models\Category;
use App\Models\Food;
use App\Models\Variant;
use App\Models\VariantItem;
use App\Rules\FileTypeValidate;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class FoodController extends Controller
{
    public function index()
    {
        $pageTitle = 'All Foods';
        $foods     = $this->filterFoods();
        return view('Template::restaurant.food.index', compact('pageTitle', 'foods'));
    }

    public function published()
    {
        $pageTitle = 'Published Foods';
        $foods     = $this->filterFoods('published');
        return view('Template::restaurant.food.index', compact('pageTitle', 'foods'));
    }

    public function draft()
    {
        $pageTitle = 'Draft Foods';
        $foods     = $this->filterFoods('draft');
        return view('Template::restaurant.food.index', compact('pageTitle', 'foods'));
    }
    protected function filterFoods($scope = null)
    {
        $foods     = Food::where('restaurant_id', authRestaurantId());
        if ($scope) {
            $foods = $foods->$scope();
        }
        return $foods->orderBy('id', 'desc')->searchable(['name'])->paginate(getPaginate());
    }

    public function status(Request $request, $id)
    {
        $food = Food::where('id', $id)->where('restaurant_id', authRestaurantId())->first();

        if (!$food) {
            return returnResponse('Food not found');
        }

        if ($food->status == Status::FOOD_PUBLISHED) {
            $food->status     = Status::FOOD_DRAFT;
            $food->step       = 4;
            $notify[] = ['success', 'Food is drafted'];
        } else {
            if ($food->step < 4) {
                return returnBack('Fill up necessary steps', route: 'restaurant.food.index');
            }
            $food->status     = Status::FOOD_PUBLISHED;
            $food->step       = 5;
            $notify[] = ['success', 'Food is published'];
        }

        $food->save();
        return back()->withNotify($notify);
    }

    public function overview($id = 0)
    {
        $pageTitle    = 'Food Overview';
        $categories   = Category::where('restaurant_id', authRestaurantId())->orWhere('restaurant_id', Status::ADMIN_CATEGORIES)->active()->orderBy('sort_order')->get();
        $food         = Food::where('id', $id)->where('restaurant_id', authRestaurantId())->first();
        if (!$food && $id) {
            return returnBack('Food Not Found', route: 'restaurant.food.index');
        }

        if ($categories->count() < 1) {
            return returnBack('No Categories Found', route: 'restaurant.categories');
        }

        if ($food) {
            $notify[] = ['warning', 'Please fill out all the information'];

            if ($food->step == 1 && $food->status == Status::FOOD_DRAFT) {
                return to_route('restaurant.food.image', $food->id)->withNotify($notify);
            }
            if ($food->step == 2 && $food->status == Status::FOOD_DRAFT) {
                return to_route('restaurant.food.variation', $food->id)->withNotify($notify);
            }
            if ($food->step == 3 && $food->status == Status::FOOD_DRAFT) {
                return to_route('restaurant.food.addons', $food->id)->withNotify($notify);
            }
            if ($food->step == 4 && $food->status == Status::FOOD_DRAFT) {
                return to_route('restaurant.food.publish', $food->id)->withNotify($notify);
            }
        }

        return view('Template::restaurant.food.overview', compact('pageTitle', 'categories', 'food'));
    }

    public function storeOverview(Request $request, $id = 0)
    {
        $validation             = Validator::make($request->all(), [
            'name'              => 'required|string',
            'short_description' => 'required|string',
            'categories'        => 'required|array',
            'categories.*'      => 'exists:categories,id',
            'price'             => 'required|numeric|gt:0',
        ]);

        if ($validation->fails()) {
            return returnResponse($validation->errors()->all());
        }

        foreach ($request->categories as $category) {
            $category = Category::where('id', $category)->active()->first();
            if (!$category) {
                return returnResponse('Category not found');
            }
        }

        $restaurant = authRestaurant();

        if ($id) {
            $food = Food::where('restaurant_id', $restaurant->id)->where('id', $id)->first();
            if (!$food) {
                return returnResponse('Food not found');
            }
        } else {
            $food = new Food();
        }

        $isUpdate = $food->step > 1;

        if (!$isUpdate) {
            $food->status     = Status::FOOD_DRAFT;
            $food->step       = 1;
            $food->save();
            $food->categories()->attach($request->categories);
        } else {
            $food->status     = Status::FOOD_PUBLISHED;
            $food->categories()->detach();
            $food->categories()->attach($request->categories);
        }

        $food->restaurant_id      = authRestaurantId();
        $food->name               = $request->name;
        $food->short_description  = $request->short_description;
        $food->price              = $request->price;
        $food->save();

        return $this->returnResponse('restaurant.food.image', $food);
    }

    public function image($id)
    {
        $pageTitle = 'Food Image';
        $food      = Food::where('id', $id)->where('restaurant_id', authRestaurantId())->firstOrFail();
        abort_if($food->step < 1, 404);
        return view('Template::restaurant.food.image', compact('pageTitle', 'food'));
    }

    public function storeImage(Request $request, $id = 0)
    {
        $validation = Validator::make($request->all(), [
            'cover_image.*'    => ['nullable', 'image', new FileTypeValidate(['jpg', 'jpeg', 'png'])],
        ]);

        if ($validation->fails()) {
            return returnResponse($validation->errors()->all());
        }

        $restaurant = authRestaurant();

        if ($id) {
            $food = Food::where('restaurant_id', $restaurant->id)->where('id', $id)->first();
            if (!$food) {
                return returnResponse('Food not found');
            }
        } else {
            $food = new Food();
        }

        $isUpdate = $food->step > 1;

        //if cover image exists in request
        if (@$request->image[0]) {
            $food->image = fileUploader(
                $request->image[0],
                getFilePath('food'),
                getFileSize('food'),
                $old = ($isUpdate ? $food->image : null),
                $thumb = getFileThumbSize('food')
            );
        }

        if (!$isUpdate) {
            $food->status     = Status::FOOD_DRAFT;
            $food->step       = 2;
        } else {
            $food->status     = Status::FOOD_PUBLISHED;
        }

        $food->save();

        return $this->returnResponse('restaurant.food.variation', $food, $isUpdate);
    }


    public function variation($id)
    {
        $pageTitle = 'Food Variations';
        $food      = Food::where('id', $id)->where('restaurant_id', authRestaurantId())->with('variations', 'variations.items')->firstOrFail();
        abort_if($food->step < 2, 404);
        return view('Template::restaurant.food.variation', compact('pageTitle', 'food'));
    }

    public function storeVariation(Request $request, $id)
    {
        $validation = Validator::make($request->all(), [
            'variations'                 => 'nullable|array',
            'variations.*.title'         => 'required|string',
            'variations.*.items'         => 'required|array',
            'variations.*.items.*.name'  => 'required|string',
            'variations.*.items.*.price' => 'required|numeric|gte:0',
        ], [
            'variations.*.title.required'         => 'Please provide a title for variation',
            'variations.*.items.required'         => 'Please provide variation items',
            'variations.*.items.*.name.required'  => 'Please provide a name for item',
            'variations.*.items.*.price.required' => 'Please provide a price for item',
            'variations.*.items.*.price.gt'       => 'Item price must be greater than 0',
        ]);

        if ($validation->fails()) {
            return returnResponse($validation->errors()->all());
        }

        $restaurant = authRestaurant();
        $food = Food::where('restaurant_id', $restaurant->id)->where('id', $id)->first();
        if (!$food) {
            return returnResponse('Food not found');
        }

        $isUpdate = $food->step > 2;

        // Delete old data
        $variants = Variant::where('food_id', $food->id)->get();
        foreach ($variants as $variant) {
            foreach ($variant->items as $item) {
                $item->delete(); // Delete each item
            }
            $variant->delete(); // Delete the variant
        }

        //Insert new data
        $variations = $request->variations;
        if ($variations) {
            foreach ($variations as $data) {
                $variant          = new Variant();
                $variant->food_id = $food->id;
                $variant->title   = $data['title'];
                $variant->save();

                foreach ($data['items'] as $item) {
                    $variantItem             = new VariantItem();
                    $variantItem->variant_id = $variant->id;
                    $variantItem->name       = $item['name'];
                    $variantItem->price      = $item['price'];
                    $variantItem->save();
                }
            }
        }

        if (!$isUpdate) {
            $food->status     = Status::FOOD_DRAFT;
            $food->step       = 3;
        } else {
            $food->status     = Status::FOOD_PUBLISHED;
        }

        $food->save();

        return $this->returnResponse('restaurant.food.addons', $food, $isUpdate);
    }

    public function addons($id)
    {
        $pageTitle = 'Food Addons';
        $food      = Food::where('id', $id)->where('restaurant_id', authRestaurantId())->with('addons')->firstOrFail();
        abort_if($food->step < 3, 404);
        return view('Template::restaurant.food.addons', compact('pageTitle', 'food'));
    }

    public function storeAddons(Request $request, $id)
    {
        $validation = Validator::make($request->all(), [
            'addons'         => 'nullable|array',
            'addons.*.name'  => 'required|string',
            'addons.*.price' => 'required|numeric|gt:0',
        ], [
            'addons.*.name.required'  => 'Please insert addon name',
            'addons.*.price.required' => 'Please insert addon price',
            'addons.*.price.gt'       => 'Addon price must be greater than 0',
        ]);

        if ($validation->fails()) {
            return returnResponse($validation->errors()->all());
        }

        $restaurant = authRestaurant();

        $food = Food::where('restaurant_id', $restaurant->id)->where('id', $id)->first();
        if (!$food) {
            return returnResponse('Food not found');
        }

        $isUpdate = $food->step > 3;

        // Delete old data
        $addons = Addon::where('food_id', $food->id)->get();
        foreach ($addons as $addon) {
            $addon->delete();
        }

        $addons = $request->addons;

        //Add new data
        if ($addons) {
            foreach ($addons as $data) {
                $addon             = new Addon();
                $addon->food_id    = $food->id;
                $addon->name       = $data["name"];
                $addon->price      = $data["price"];
                $addon->save();
            }
        }

        if (!$isUpdate) {
            $food->status     = Status::FOOD_DRAFT;
            $food->step       = 4;
        } else {
            $food->status     = Status::FOOD_PUBLISHED;
        }

        $food->save();

        return $this->returnResponse('restaurant.food.publish', $food, $isUpdate);
    }

    public function publish($id)
    {
        $pageTitle = 'Publish Food';
        $food      = Food::where('id', $id)->where('restaurant_id', authRestaurantId())->firstOrFail();
        abort_if($food->step < 4, 404);
        return view('Template::restaurant.food.publish', compact('pageTitle', 'food'));
    }

    public function storePublish(Request $request, $id)
    {
        $food = Food::where('id', $id)->where('restaurant_id', authRestaurantId())->first();

        if (!$food) {
            return returnResponse('Food not found');
        }

        if ($request->is_published == Status::NO) {
            //draft
            $food->status     = Status::FOOD_DRAFT;
            $food->step       = 4;
        } else {
            //published
            $food->status     = Status::FOOD_PUBLISHED;
            $food->step       = 5;
        }

        $food->save();

        //TODO: send notification to admin

        return response()->json([
            'success'      => true,
            'status'       => $food->is_published,
            'redirect_url' => route('restaurant.food.index')
        ]);
    }

    private function returnResponse($route, $food, $isUpdate = null)
    {
        $arr = [
            'success'      => true,
            'redirect_url' => route($route, $food->id)
        ];

        if ($isUpdate != null) {
            $arr['is_update'] = $isUpdate;
        }

        return response()->json($arr);
    }
}
