<?php

namespace App\Http\Controllers\Restaurant\Auth;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Lib\Intended;
use App\Models\AdminNotification;
use App\Models\Location;
use App\Models\Restaurant;
use App\Models\UserLogin;
use Illuminate\Auth\Events\Registered;
use Illuminate\Foundation\Auth\RegistersUsers;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rules\Password;

class RegisterController extends Controller
{

    use RegistersUsers;

    public function __construct()
    {
        parent::__construct();
    }

    public function showRegistrationForm()
    {
        $pageTitle = "Restaurant Register";
        $locations = Location::active()->orderBy('sort_order')->get();
        Intended::identifyRoute();
        return view('Template::restaurant.auth.register', compact('pageTitle', 'locations'));
    }


    protected function validator(array $data)
    {

        $passwordValidation = Password::min(6);

        if (gs('secure_password')) {
            $passwordValidation = $passwordValidation->mixedCase()->numbers()->symbols()->uncompromised();
        }

        $agree = 'nullable';
        if (gs('agree')) {
            $agree = 'required';
        }

        $validate     = Validator::make($data, [
            'email'     => 'required|string|email|unique:restaurants',
            'password'  => ['required', 'confirmed', $passwordValidation],
            'captcha'   => 'sometimes|required',
            'agree'     => $agree
        ]);

        return $validate;
    }

    public function register(Request $request)
    {
        if (!gs('restaurant_registration')) {
            $notify[] = ['error', 'Registration not allowed'];
            return back()->withNotify($notify);
        }
        $this->validator($request->all())->validate();

        $request->session()->regenerateToken();

        if (!verifyCaptcha()) {
            $notify[] = ['error', 'Invalid captcha provided'];
            return back()->withNotify($notify);
        }


        $exist = Restaurant::where('mobile', $request->mobile)->first();
        if ($exist) {
            $notify[] = ['error', 'The mobile number already exists'];
            return back()->withNotify($notify)->withInput();
        }

        $location = Location::active()->find($request->location);
        if (!$location) {
            $notify[] = ['error', 'The location does not exist'];
            return back()->withNotify($notify)->withInput();
        }

        event(new Registered($restaurant = $this->create($request->all())));

        $this->guard()->login($restaurant);

        return $this->registered($request, $restaurant)
            ?: redirect($this->redirectPath());
    }



    protected function create(array $data)
    {
        //Restaurant Create
        $restaurant            = new Restaurant();
        $restaurant->email     = strtolower($data['email']);
        $restaurant->password  = Hash::make($data['password']);
        $restaurant->kv        = gs('kv') ? Status::NO : Status::YES;
        $restaurant->ev        = gs('ev') ? Status::NO : Status::YES;
        $restaurant->sv        = gs('sv') ? Status::NO : Status::YES;
        $restaurant->ts        = Status::DISABLE;
        $restaurant->tv        = Status::ENABLE;
        $restaurant->mobile    = $data['mobile'];
        $restaurant->location_id    = $data['location'];
        $restaurant->save();

        $adminNotification                  = new AdminNotification();
        $adminNotification->restaurant_id   = $restaurant->id;
        $adminNotification->title           = 'New member registered';
        $adminNotification->click_url       = urlPath('admin.restaurants.detail', $restaurant->id);
        $adminNotification->save();


        //Login Log Create
        $ip              = getRealIP();
        $exist           = UserLogin::where('user_ip', $ip)->first();
        $restaurantLogin = new UserLogin();

        if ($exist) {
            $restaurantLogin->longitude    = $exist->longitude;
            $restaurantLogin->latitude     = $exist->latitude;
            $restaurantLogin->city         = $exist->city;
            $restaurantLogin->country_code = $exist->country_code;
            $restaurantLogin->country      = $exist->country;
        } else {
            $info                          = json_decode(json_encode(getIpInfo()), true);
            $restaurantLogin->longitude    = @implode(',', $info['long']);
            $restaurantLogin->latitude     = @implode(',', $info['lat']);
            $restaurantLogin->city         = @implode(',', $info['city']);
            $restaurantLogin->country_code = @implode(',', $info['code']);
            $restaurantLogin->country      = @implode(',', $info['country']);
        }

        $restaurantAgent                = osBrowser();
        $restaurantLogin->restaurant_id = $restaurant->id;
        $restaurantLogin->user_ip = $ip;

        $restaurantLogin->browser = @$restaurantAgent['browser'];
        $restaurantLogin->os      = @$restaurantAgent['os_platform'];
        $restaurantLogin->save();

        if (auth()->check()) {
            auth()->logout();
        }

        return $restaurant;
    }

    public function checkUser(Request $request)
    {
        $exist['data'] = false;
        $exist['type'] = null;
        if ($request->email) {
            $exist['data'] = Restaurant::where('email', $request->email)->exists();
            $exist['type'] = 'email';
            $exist['field'] = 'Email';
        }
        if ($request->mobile) {
            $exist['data'] = Restaurant::where('mobile', $request->mobile)->where('dial_code', $request->mobile_code)->exists();
            $exist['type'] = 'mobile';
            $exist['field'] = 'Mobile';
        }
        if ($request->username) {
            $exist['data'] = Restaurant::where('username', $request->username)->exists();
            $exist['type'] = 'username';
            $exist['field'] = 'Username';
        }
        return response($exist);
    }

    protected function guard()
    {
        return auth()->guard('restaurant');
    }

    public function registered()
    {
        return to_route('restaurant.home');
    }
}
