<?php

namespace App\Http\Controllers\Gateway;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Lib\FormProcessor;
use App\Models\AdminNotification;
use App\Models\Deposit;
use App\Models\GatewayCurrency;
use App\Models\Order;
use App\Models\Transaction;
use App\Models\User;
use Illuminate\Http\Request;

class PaymentController extends Controller
{
    public function deposit($orderId = 0)
    {
        $gatewayCurrency = GatewayCurrency::whereHas('method', function ($gate) {
            $gate->where('status', Status::ENABLE);
        })->with('method')->orderby('name')->get();
        $pageTitle = 'Deposit Methods';

        $order = null;
        if ($orderId) {
            $pageTitle = 'Payment Methods';
            $order = Order::userOrders()->paymentPending()->find($orderId);
            if (!$order) {
                $notify[] = ['error', 'You can not make payment for this order'];
                return to_route('user.home')->withNotify($notify);
            }
        }

        $type = authenticatedUser()['type'];

        return view("Template::$type.payment.deposit", compact('gatewayCurrency', 'pageTitle', 'order'));
    }

    public function depositInsert(Request $request, $orderId = 0)
    {
        $request->validate([
            'amount' => 'required|numeric|gt:0',
            'gateway' => 'required',
            'currency' => 'required',
        ]);

        if ($orderId) {
            $order = Order::userOrders()->paymentPending()->find($orderId);
            $request->amount = $order->total_price;

            if (!$order) {
                $notify[] = ['error', 'Unauthorized action'];
                return to_route('user.order.details', $order->id)->withNotify($notify);
            }

            if ($request->gateway == "wallet") {
                $user = $order->user;
                if ($order->total_price > $user->balance) {
                    $notify[] = ['error', 'Insufficient Balance'];
                    return to_route('user.order.details', $order->id)->withNotify($notify);
                }

                self::confirmOrder($order);

                $notify[] = ['success', 'Payment done successfully'];
                return to_route('user.order.details', $order->id)->withNotify($notify);
            }
        }


        $user = auth()->user();
        $gate = GatewayCurrency::whereHas('method', function ($gate) {
            $gate->where('status', Status::ENABLE);
        })->where('method_code', $request->gateway)->where('currency', $request->currency)->first();
        if (!$gate) {
            $notify[] = ['error', 'Invalid gateway'];
            return back()->withNotify($notify);
        }

        if ($gate->min_amount > $request->amount || $gate->max_amount < $request->amount) {
            $notify[] = ['error', 'Please follow deposit limit'];
            return back()->withNotify($notify);
        }

        $charge = $gate->fixed_charge + ($request->amount * $gate->percent_charge / 100);
        $payable = $request->amount + $charge;
        $finalAmount = $payable * $gate->rate;

        $type = authenticatedUser()['type'];
        $typeCol = $type . "_id";

        $deposit = new Deposit();
        $deposit->$typeCol = authenticatedUser()['user']->id;
        if ($orderId) {
            $deposit->order_id = $orderId;
        }
        $deposit->method_code = $gate->method_code;
        $deposit->method_currency = strtoupper($gate->currency);
        $deposit->amount = $request->amount;
        $deposit->charge = $charge;
        $deposit->rate = $gate->rate;
        $deposit->final_amount = $finalAmount;
        $deposit->btc_amount = 0;
        $deposit->btc_wallet = "";
        $deposit->trx = getTrx();
        $deposit->success_url = urlPath("$type.deposit.history");
        if ($orderId) {
            $deposit->success_url = urlPath('user.order.details', $orderId);
        }
        $deposit->failed_url = urlPath("$type.deposit.history");
        $deposit->save();
        session()->put('Track', $deposit->trx);
        return to_route($type . '.deposit.confirm');
    }


    public function appDepositConfirm($hash)
    {
        try {
            $id = decrypt($hash);
        } catch (\Exception $ex) {
            abort(404);
        }
        $data = Deposit::where('id', $id)->where('status', Status::PAYMENT_INITIATE)->orderBy('id', 'DESC')->firstOrFail();
        $user = User::findOrFail($data->user_id);
        auth()->login($user);
        session()->put('Track', $data->trx);
        return to_route('user.deposit.confirm');
    }


    public function depositConfirm()
    {
        $track = session()->get('Track');
        $deposit = Deposit::where('trx', $track)->where('status', Status::PAYMENT_INITIATE)->orderBy('id', 'DESC')->with('gateway')->firstOrFail();


        $type = $deposit->user_id ? 'user' : 'restaurant';
        $user = $deposit->user ?? $deposit->restaurant;

        if ($deposit->method_code >= 1000) {
            return to_route($type . '.deposit.manual.confirm');
        }


        $dirName = $deposit->gateway->alias;
        $new = __NAMESPACE__ . '\\' . $dirName . '\\ProcessController';

        $data = $new::process($deposit, $user, $type);
        $data = json_decode($data);



        if (isset($data->error)) {
            $notify[] = ['error', $data->message];
            return back()->withNotify($notify);
        }
        if (isset($data->redirect)) {
            return redirect($data->redirect_url);
        }

        // for Stripe V3
        if (@$data->session) {
            $deposit->btc_wallet = $data->session->id;
            $deposit->save();
        }

        $pageTitle = $deposit->order_id ? 'Order Payment Confirm' : 'Payment Confirm';
        return view("Template::$data->view", compact('data', 'pageTitle', 'deposit'));
    }


    public static function userDataUpdate($deposit, $isManual = null)
    {
        if ($deposit->status == Status::PAYMENT_INITIATE || $deposit->status == Status::PAYMENT_PENDING) {
            $deposit->status = Status::PAYMENT_SUCCESS;
            $deposit->save();

            $user    = $deposit->user ?? $deposit->restaurant;
            $type    = $deposit->user ? 'user' : 'restaurant';
            $typeCol = $type . '_id';

            $user->balance += $deposit->amount;
            $user->save();

            $methodName = $deposit->methodName();

            $transaction = new Transaction();
            $transaction->user_id = $deposit->user_id;
            $transaction->amount = $deposit->amount;
            $transaction->post_balance = $user->balance;
            $transaction->charge = $deposit->charge;
            $transaction->trx_type = '+';
            $transaction->details = 'Deposit Via ' . $methodName;
            $transaction->trx = $deposit->trx;
            $transaction->remark = 'deposit';
            $transaction->save();

            if (!$isManual) {
                $adminNotification = new AdminNotification();
                $adminNotification->user_id = $user->id;
                $adminNotification->title = 'Deposit successful via ' . $methodName;
                $adminNotification->click_url = urlPath('admin.deposit.successful');
                $adminNotification->save();
            }

            notify($user, $isManual ? 'DEPOSIT_APPROVE' : 'DEPOSIT_COMPLETE', [
                'method_name' => $methodName,
                'method_currency' => $deposit->method_currency,
                'method_amount' => showAmount($deposit->final_amount, currencyFormat: false),
                'amount' => showAmount($deposit->amount, currencyFormat: false),
                'charge' => showAmount($deposit->charge, currencyFormat: false),
                'rate' => showAmount($deposit->rate, currencyFormat: false),
                'trx' => $deposit->trx,
                'post_balance' => showAmount($user->balance)
            ]);

            //substract balance and make payment
            if ($deposit->order_id != 0 && $deposit->user) {
                $order = $deposit->order;
                self::confirmOrder($order);
            }
        }
    }

    public function manualDepositConfirm()
    {
        $track = session()->get('Track');
        $data = Deposit::with('gateway')->where('status', Status::PAYMENT_INITIATE)->where('trx', $track)->first();
        abort_if(!$data, 404);
        if ($data->method_code > 999) {
            $pageTitle = 'Confirm Deposit';
            $method = $data->gatewayCurrency();
            $gateway = $method->method;
            $type      = $data->user ? 'user' : 'restaurant';
            return view("Template::$type.payment.manual", compact('data', 'pageTitle', 'method', 'gateway'));
        }
        abort(404);
    }

    public function manualDepositUpdate(Request $request)
    {
        $track = session()->get('Track');
        $data = Deposit::with('gateway')->where('status', Status::PAYMENT_INITIATE)->where('trx', $track)->first();
        abort_if(!$data, 404);
        $gatewayCurrency = $data->gatewayCurrency();
        $gateway = $gatewayCurrency->method;
        $formData = $gateway->form->form_data;

        $formProcessor = new FormProcessor();
        $validationRule = $formProcessor->valueValidation($formData);
        $request->validate($validationRule);
        $userData = $formProcessor->processFormData($request, $formData);


        $data->detail = $userData;
        $data->status = Status::PAYMENT_PENDING;
        $data->save();

        $type    = $data->user ? 'user' : 'restaurant';
        $typeCol = $type . '_id';

        $adminNotification = new AdminNotification();
        $adminNotification->$typeCol = $data->$type->id;
        $adminNotification->title = 'Deposit request from ' .  $data->$type->username;
        $adminNotification->click_url = urlPath('admin.deposit.details', $data->id);
        $adminNotification->save();

        notify($data->$type, 'DEPOSIT_REQUEST', [
            'method_name' => $data->gatewayCurrency()->name,
            'method_currency' => $data->method_currency,
            'method_amount' => showAmount($data->final_amount, currencyFormat: false),
            'amount' => showAmount($data->amount, currencyFormat: false),
            'charge' => showAmount($data->charge, currencyFormat: false),
            'rate' => showAmount($data->rate, currencyFormat: false),
            'trx' => $data->trx
        ]);

        $notify[] = ['success', 'You have deposit request has been taken'];
        return to_route($type . '.deposit.history')->withNotify($notify);
    }

    public static function confirmOrder($order)
    {
        $restaurantName = $order->restaurant->restaurant_name;
        $user = $order->user;
        $user->balance -= $order->total_price;
        $user->save();

        $transaction = new Transaction();
        $transaction->user_id = $order->user_id;
        $transaction->amount = $order->total_price;
        $transaction->post_balance = $user->balance;
        $transaction->charge = 0;
        $transaction->trx_type = '-';
        $transaction->details = "Payment Completed for order from restaurant " . $restaurantName;
        $transaction->trx = getTrx();
        $transaction->remark = 'order_payment';
        $transaction->save();

        //change order status
        $order->payment_status = Status::PAID; //order payment done
        $order->status = Status::ORDER_PROCESSING; //order processing
        $order->save();

        notifyUser('user', 'ORDER_PROCESSING_USER', $order, $transaction->trx);
        notifyUser('restaurant', 'ORDER_PROCESSING_RESTAURANT', $order, $transaction->trx);
    }
}
