<?php

namespace App\Http\Controllers\Admin;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Models\Deposit;
use App\Models\Location;
use App\Models\NotificationLog;
use App\Models\NotificationTemplate;
use App\Models\Transaction;
use App\Models\Restaurant;
use App\Models\Withdrawal;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Rules\FileTypeValidate;

class ManageRestaurantsController extends Controller
{

    public function allRestaurants()
    {
        $pageTitle = 'All Restaurants';
        $restaurants = $this->restaurantData();
        return view('admin.restaurants.list', compact('pageTitle', 'restaurants'));
    }

    public function activeRestaurants()
    {
        $pageTitle = 'Active Restaurants';
        $restaurants = $this->restaurantData('active');
        return view('admin.restaurants.list', compact('pageTitle', 'restaurants'));
    }

    public function bannedRestaurants()
    {
        $pageTitle = 'Banned Restaurants';
        $restaurants = $this->restaurantData('banned');
        return view('admin.restaurants.list', compact('pageTitle', 'restaurants'));
    }

    public function emailUnverifiedRestaurants()
    {
        $pageTitle = 'Email Unverified Restaurants';
        $restaurants = $this->restaurantData('emailUnverified');
        return view('admin.restaurants.list', compact('pageTitle', 'restaurants'));
    }

    public function kycUnverifiedRestaurants()
    {
        $pageTitle = 'KYC Unverified Restaurants';
        $restaurants = $this->restaurantData('kycUnverified');
        return view('admin.restaurants.list', compact('pageTitle', 'restaurants'));
    }

    public function kycPendingRestaurants()
    {
        $pageTitle = 'KYC Pending Restaurants';
        $restaurants = $this->restaurantData('kycPending');
        return view('admin.restaurants.list', compact('pageTitle', 'restaurants'));
    }

    public function emailVerifiedRestaurants()
    {
        $pageTitle = 'Email Verified Restaurants';
        $restaurants = $this->restaurantData('emailVerified');
        return view('admin.restaurants.list', compact('pageTitle', 'restaurants'));
    }


    public function mobileUnverifiedRestaurants()
    {
        $pageTitle = 'Mobile Unverified Restaurants';
        $restaurants = $this->restaurantData('mobileUnverified');
        return view('admin.restaurants.list', compact('pageTitle', 'restaurants'));
    }


    public function mobileVerifiedRestaurants()
    {
        $pageTitle = 'Mobile Verified Restaurants';
        $restaurants = $this->restaurantData('mobileVerified');
        return view('admin.restaurants.list', compact('pageTitle', 'restaurants'));
    }


    public function restaurantsWithBalance()
    {
        $pageTitle = 'Restaurants with Balance';
        $restaurants = $this->restaurantData('withBalance');
        return view('admin.restaurants.list', compact('pageTitle', 'restaurants'));
    }


    protected function restaurantData($scope = null)
    {
        if ($scope) {
            $restaurants = Restaurant::$scope();
        } else {
            $restaurants = Restaurant::query();
        }
        return $restaurants->searchable(['username', 'email'])->orderBy('id', 'desc')->with('location')->withCount('foods')->paginate(getPaginate());
    }


    public function detail($id)
    {
        $restaurant = Restaurant::findOrFail($id);
        $pageTitle = 'Restaurant Detail - ' . $restaurant->username;


        $totalDeposit = Deposit::where('restaurant_id', $restaurant->id)->successful()->sum('amount');
        $totalWithdrawals = Withdrawal::where('restaurant_id', $restaurant->id)->approved()->sum('amount');
        $totalTransaction = Transaction::where('restaurant_id', $restaurant->id)->count();
        $locations       = Location::active()->get();
        return view('admin.restaurants.detail', compact('pageTitle', 'restaurant', 'totalDeposit', 'totalWithdrawals', 'totalTransaction', 'locations'));
    }


    public function kycDetails($id)
    {
        $pageTitle = 'KYC Details';
        $restaurant = Restaurant::findOrFail($id);
        return view('admin.restaurants.kyc_detail', compact('pageTitle', 'restaurant'));
    }

    public function kycApprove($id)
    {
        $restaurant = Restaurant::findOrFail($id);
        $restaurant->kv = Status::KYC_VERIFIED;
        $restaurant->save();

        notify($restaurant, 'KYC_APPROVE', []);

        $notify[] = ['success', 'KYC approved successfully'];
        return to_route('admin.restaurants.kyc.pending')->withNotify($notify);
    }

    public function kycReject(Request $request, $id)
    {
        $request->validate([
            'reason' => 'required'
        ]);
        $restaurant = Restaurant::findOrFail($id);
        $restaurant->kv = Status::KYC_UNVERIFIED;
        $restaurant->kyc_rejection_reason = $request->reason;
        $restaurant->save();

        notify($restaurant, 'KYC_REJECT', [
            'reason' => $request->reason
        ]);

        $notify[] = ['success', 'KYC rejected successfully'];
        return to_route('admin.restaurants.kyc.pending')->withNotify($notify);
    }


    public function update(Request $request, $id)
    {
        $restaurant = Restaurant::findOrFail($id);
        $request->validate([
            'email'     => 'required|email|string|max:40|unique:restaurants,email,' . $restaurant->id,
            'mobile'    => 'required|string|max:40',
        ]);

        $exists = Restaurant::where('mobile', $request->mobile)->where('id', '!=', $restaurant->id)->exists();
        if ($exists) {
            $notify[] = ['error', 'The mobile number already exists.'];
            return back()->withNotify($notify);
        }

        $restaurant->location_id = $request->location;
        $restaurant->mobile      = $request->mobile;

        $restaurant->email       = $request->email;

        $restaurant->address = $request->address;
        $restaurant->city    = $request->city;
        $restaurant->state   = $request->state;
        $restaurant->zip     = $request->zip;

        $restaurant->ev = $request->ev ? Status::VERIFIED : Status::UNVERIFIED;
        $restaurant->sv = $request->sv ? Status::VERIFIED : Status::UNVERIFIED;
        $restaurant->ts = $request->ts ? Status::ENABLE : Status::DISABLE;
        if (!$request->kv) {
            $restaurant->kv = Status::KYC_UNVERIFIED;
            if ($restaurant->kyc_data) {
                foreach ($restaurant->kyc_data as $kycData) {
                    if ($kycData->type == 'file') {
                        fileManager()->removeFile(getFilePath('verify') . '/' . $kycData->value);
                    }
                }
            }
            $restaurant->kyc_data = null;
        } else {
            $restaurant->kv = Status::KYC_VERIFIED;
        }
        $restaurant->save();

        $notify[] = ['success', 'Restaurant details updated successfully'];
        return back()->withNotify($notify);
    }

    public function addSubBalance(Request $request, $id)
    {
        $request->validate([
            'amount' => 'required|numeric|gt:0',
            'act' => 'required|in:add,sub',
            'remark' => 'required|string|max:255',
        ]);

        $restaurant = Restaurant::findOrFail($id);
        $amount = $request->amount;
        $trx = getTrx();

        $transaction = new Transaction();

        if ($request->act == 'add') {
            $restaurant->balance += $amount;

            $transaction->trx_type = '+';
            $transaction->remark = 'balance_add';

            $notifyTemplate = 'BAL_ADD';

            $notify[] = ['success', 'Balance added successfully'];
        } else {
            if ($amount > $restaurant->balance) {
                $notify[] = ['error', $restaurant->username . ' doesn\'t have sufficient balance.'];
                return back()->withNotify($notify);
            }

            $restaurant->balance -= $amount;

            $transaction->trx_type = '-';
            $transaction->remark = 'balance_subtract';

            $notifyTemplate = 'BAL_SUB';
            $notify[] = ['success', 'Balance subtracted successfully'];
        }

        $restaurant->save();

        $transaction->restaurant_id = $restaurant->id;
        $transaction->amount = $amount;
        $transaction->post_balance = $restaurant->balance;
        $transaction->charge = 0;
        $transaction->trx =  $trx;
        $transaction->details = $request->remark;
        $transaction->save();

        notify($restaurant, $notifyTemplate, [
            'trx' => $trx,
            'amount' => showAmount($amount, currencyFormat: false),
            'remark' => $request->remark,
            'post_balance' => showAmount($restaurant->balance, currencyFormat: false)
        ]);

        return back()->withNotify($notify);
    }

    public function login($id)
    {
        if (Auth::check()) {
            Auth::logout();
        }
        Auth::guard('restaurant')->loginUsingId($id);
        return to_route('restaurant.home');
    }

    public function status(Request $request, $id)
    {
        $restaurant = Restaurant::findOrFail($id);
        if ($restaurant->status == Status::RESTAURANT_ACTIVE) {
            $request->validate([
                'reason' => 'required|string|max:255'
            ]);
            $restaurant->status = Status::RESTAURANT_BAN;
            $restaurant->ban_reason = $request->reason;
            $notify[] = ['success', 'Restaurant banned successfully'];
        } else {
            $restaurant->status = Status::RESTAURANT_ACTIVE;
            $restaurant->ban_reason = null;
            $notify[] = ['success', 'Restaurant unbanned successfully'];
        }
        $restaurant->save();
        return back()->withNotify($notify);
    }


    public function showNotificationSingleForm($id)
    {
        $restaurant = Restaurant::findOrFail($id);
        if (!gs('en') && !gs('sn') && !gs('pn')) {
            $notify[] = ['warning', 'Notification options are disabled currently'];
            return to_route('admin.restaurants.detail', $restaurant->id)->withNotify($notify);
        }
        $pageTitle = 'Send Notification to ' . $restaurant->username;
        return view('admin.restaurants.notification_single', compact('pageTitle', 'restaurant'));
    }

    public function sendNotificationSingle(Request $request, $id)
    {
        $request->validate([
            'message' => 'required',
            'via'     => 'required|in:email,sms,push',
            'subject' => 'required_if:via,email,push',
            'image'   => ['nullable', 'image', new FileTypeValidate(['jpg', 'jpeg', 'png'])],
        ]);

        if (!gs('en') && !gs('sn') && !gs('pn')) {
            $notify[] = ['warning', 'Notification options are disabled currently'];
            return to_route('admin.dashboard')->withNotify($notify);
        }

        $imageUrl = null;
        if ($request->via == 'push' && $request->hasFile('image')) {
            $imageUrl = fileUploader($request->image, getFilePath('push'));
        }

        $template = NotificationTemplate::where('act', 'DEFAULT')->where($request->via . '_status', Status::ENABLE)->exists();
        if (!$template) {
            $notify[] = ['warning', 'Default notification template is not enabled'];
            return back()->withNotify($notify);
        }

        $restaurant = Restaurant::findOrFail($id);
        notify($restaurant, 'DEFAULT', [
            'subject' => $request->subject,
            'message' => $request->message,
        ], [$request->via], pushImage: $imageUrl);
        $notify[] = ['success', 'Notification sent successfully'];
        return back()->withNotify($notify);
    }

    public function showNotificationAllForm()
    {
        if (!gs('en') && !gs('sn') && !gs('pn')) {
            $notify[] = ['warning', 'Notification options are disabled currently'];
            return to_route('admin.dashboard')->withNotify($notify);
        }

        $notifyToRestaurant = Restaurant::notifyToRestaurant();
        $restaurants        = Restaurant::active()->count();
        $pageTitle    = 'Notification to Verified Restaurants';

        if (session()->has('SEND_NOTIFICATION') && !request()->email_sent) {
            session()->forget('SEND_NOTIFICATION');
        }

        return view('admin.restaurants.notification_all', compact('pageTitle', 'restaurants', 'notifyToRestaurant'));
    }

    public function sendNotificationAll(Request $request)
    {
        $request->validate([
            'via'                          => 'required|in:email,sms,push',
            'message'                      => 'required',
            'subject'                      => 'required_if:via,email,push',
            'start'                        => 'required|integer|gte:1',
            'batch'                        => 'required|integer|gte:1',
            'being_sent_to'                => 'required',
            'cooling_time'                 => 'required|integer|gte:1',
            'number_of_top_deposited_restaurant' => 'required_if:being_sent_to,topDepositedRestaurants|integer|gte:0',
            'number_of_days'               => 'required_if:being_sent_to,notLoginRestaurants|integer|gte:0',
            'image'                        => ["nullable", 'image', new FileTypeValidate(['jpg', 'jpeg', 'png'])],
        ], [
            'number_of_days.required_if'               => "Number of days field is required",
            'number_of_top_deposited_restaurant.required_if' => "Number of top deposited restaurant field is required",
        ]);

        if (!gs('en') && !gs('sn') && !gs('pn')) {
            $notify[] = ['warning', 'Notification options are disabled currently'];
            return to_route('admin.dashboard')->withNotify($notify);
        }


        $template = NotificationTemplate::where('act', 'DEFAULT')->where($request->via . '_status', Status::ENABLE)->exists();
        if (!$template) {
            $notify[] = ['warning', 'Default notification template is not enabled'];
            return back()->withNotify($notify);
        }

        if ($request->being_sent_to == 'selectedRestaurants') {
            if (session()->has("SEND_NOTIFICATION")) {
                $request->merge(['restaurant' => session()->get('SEND_NOTIFICATION')['restaurant']]);
            } else {
                if (!$request->restaurant || !is_array($request->restaurant) || empty($request->restaurant)) {
                    $notify[] = ['error', "Ensure that the restaurant field is populated when sending an email to the designated restaurant group"];
                    return back()->withNotify($notify);
                }
            }
        }

        $scope          = $request->being_sent_to;
        $restaurantQuery      = Restaurant::oldest()->active()->$scope();

        if (session()->has("SEND_NOTIFICATION")) {
            $totalRestaurantCount = session('SEND_NOTIFICATION')['total_restaurant'];
        } else {
            $totalRestaurantCount = (clone $restaurantQuery)->count() - ($request->start - 1);
        }


        if ($totalRestaurantCount <= 0) {
            $notify[] = ['error', "Notification recipients were not found among the selected restaurant base."];
            return back()->withNotify($notify);
        }


        $imageUrl = null;

        if ($request->via == 'push' && $request->hasFile('image')) {
            if (session()->has("SEND_NOTIFICATION")) {
                $request->merge(['image' => session()->get('SEND_NOTIFICATION')['image']]);
            }
            if ($request->hasFile("image")) {
                $imageUrl = fileUploader($request->image, getFilePath('push'));
            }
        }

        $restaurants = (clone $restaurantQuery)->skip($request->start - 1)->limit($request->batch)->get();

        foreach ($restaurants as $restaurant) {
            notify($restaurant, 'DEFAULT', [
                'subject' => $request->subject,
                'message' => $request->message,
            ], [$request->via], pushImage: $imageUrl);
        }

        return $this->sessionForNotification($totalRestaurantCount, $request);
    }


    private function sessionForNotification($totalRestaurantCount, $request)
    {
        if (session()->has('SEND_NOTIFICATION')) {
            $sessionData                = session("SEND_NOTIFICATION");
            $sessionData['total_sent'] += $sessionData['batch'];
        } else {
            $sessionData               = $request->except('_token');
            $sessionData['total_sent'] = $request->batch;
            $sessionData['total_restaurant'] = $totalRestaurantCount;
        }

        $sessionData['start'] = $sessionData['total_sent'] + 1;

        if ($sessionData['total_sent'] >= $totalRestaurantCount) {
            session()->forget("SEND_NOTIFICATION");
            $message = ucfirst($request->via) . " notifications were sent successfully";
            $url     = route("admin.restaurants.notification.all");
        } else {
            session()->put('SEND_NOTIFICATION', $sessionData);
            $message = $sessionData['total_sent'] . " " . $sessionData['via'] . "  notifications were sent successfully";
            $url     = route("admin.restaurants.notification.all") . "?email_sent=yes";
        }
        $notify[] = ['success', $message];
        return redirect($url)->withNotify($notify);
    }

    public function countBySegment($methodName)
    {
        return Restaurant::active()->$methodName()->count();
    }

    public function list()
    {
        $query = Restaurant::active();

        if (request()->search) {
            $query->where(function ($q) {
                $q->where('email', 'like', '%' . request()->search . '%')->orWhere('username', 'like', '%' . request()->search . '%');
            });
        }
        $restaurants = $query->orderBy('id', 'desc')->paginate(getPaginate());
        return response()->json([
            'success' => true,
            'restaurants'   => $restaurants,
            'more'    => $restaurants->hasMorePages()
        ]);
    }

    public function notificationLog($id)
    {
        $restaurant = Restaurant::findOrFail($id);
        $pageTitle = 'Notifications Sent to ' . $restaurant->username;
        $logs = NotificationLog::where('restaurant_id', $id)->with('restaurant')->orderBy('id', 'desc')->paginate(getPaginate());
        return view('admin.reports.notification_history', compact('pageTitle', 'logs', 'restaurant'));
    }

    public function featured($id)
    {
        $restaurant = Restaurant::findOrFail($id);
        $restaurant->is_featured = !$restaurant->is_featured;
        $restaurant->save();

        $notify[] = ['success', 'Restaurant changed successfully'];
        return back()->withNotify($notify);
    }
}
